/*
  audio.wasapi (2011-10-15)
  author: Fatbag
*/

#include "wasapi.hpp"

//Compile options
#define EXCLUSIVE_MODE
#define USE_SAFE_WASAPI_LATENCY
#define FORCE_MIN_WASAPI_LATENCY (10*10000)

#ifdef CONSOLE
  #define logerr(x) printf("%s" x "%s%u\n", "\nWASAPI: Initialization error\n", "\nError code: ", result)
#else
  #define logerr(x)
#endif

namespace ruby {

class pAudioWASAPI {
public:
  IMMDevice *pDevice;
  IAudioClient *pAudioClient;
  IAudioRenderClient *pRenderClient;
  WAVEFORMATEX *wfx;
  CRITICAL_SECTION ChangeBufferCount;
  HANDLE BufferAvailable, hThread;
  volatile bool Pause, Terminate;

  struct {
    uint64_t latency;
    unsigned bufferframes;

    uint32_t *localbuffer;
    unsigned localbuffers;
    volatile unsigned buffersfilled;
    unsigned onbuffer;
    unsigned frameoffset;
  } device;

  struct {
    bool synchronize;
    unsigned frequency;
    unsigned latency;
  } settings;

  bool cap(const string& name) const {
    if(name == Audio::Synchronize) return true;
    if(name == Audio::Frequency) return true;
    if(name == Audio::Latency) return true;
    return false;
  }

  any get(const string& name) const {
    if(name == Audio::Synchronize) return settings.synchronize;
    if(name == Audio::Frequency) return settings.frequency;
    if(name == Audio::Latency) return device.localbuffers*device.latency/10000;
    return false;
  }

  bool set(const string& name, const any& value) {
    if(name == Audio::Synchronize) {
      settings.synchronize = any_cast<bool>(value);
      return true;
    }

    if(name == Audio::Frequency) {
      settings.frequency = any_cast<unsigned>(value);
      return true;
    }

    if(name == Audio::Latency) {
      settings.latency = any_cast<unsigned>(value);
      return true;
    }

    return false;
  }

  void sample(uint16_t left, uint16_t right) {
    if(settings.synchronize) {
      WaitForSingleObject(BufferAvailable, INFINITE);
    } else if(device.buffersfilled == device.localbuffers) return; //Discard if no buffers available

    device.localbuffer[device.onbuffer*device.bufferframes + device.frameoffset] = left | (right << 16);

    if(++device.frameoffset == device.bufferframes) {
      EnterCriticalSection(&ChangeBufferCount);
      if(++device.buffersfilled == device.localbuffers) {
        ResetEvent(BufferAvailable);
      }
      LeaveCriticalSection(&ChangeBufferCount);

      device.frameoffset = 0;
      if(++device.onbuffer == device.localbuffers) device.onbuffer = 0;
    }
  }

  void clear() {
    Pause = true;
  }

  bool init() {
    BufferAvailable = CreateEvent(NULL, TRUE, FALSE, NULL);
    InitializeCriticalSection(&ChangeBufferCount);

    hThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) ThreadProc, this, 0, NULL);
    if(hThread == NULL) {
      term();
      return false;
    }

    if(WaitForSingleObject(BufferAvailable, 3000) /* Not signaled */ || Terminate) {
      term();
      return false;
    }

    SetPriority(GetCurrentThread(), false);
    return true;
  }

  void SetPriority(const HANDLE hThread, bool TimeCritical) {
    SetThreadPriority(hThread, (TimeCritical) ? THREAD_PRIORITY_TIME_CRITICAL : THREAD_PRIORITY_HIGHEST);

    HMODULE hModule = LoadLibrary(L"dwmapi.dll");
    if(hModule) {
      auto DwmEnableMMCSS = (HRESULT WINAPI (*)(BOOL)) GetProcAddress(hModule, "DwmEnableMMCSS");
      if(DwmEnableMMCSS) {
        DwmEnableMMCSS(TRUE);
      }
      FreeLibrary(hModule);
    }
    hModule = LoadLibrary(L"avrt.dll");

    if(hModule) {
      auto AvSetMmThreadCharacteristicsW =
        (HANDLE WINAPI (*)(LPCTSTR,LPDWORD)) GetProcAddress(hModule, "AvSetMmThreadCharacteristicsW");
      if(AvSetMmThreadCharacteristicsW) {
        DWORD taskindex = 0;
        HANDLE hTask = AvSetMmThreadCharacteristicsW((TimeCritical) ? L"Pro Audio" : L"Playback", &taskindex);
        auto AvSetMmThreadPriority = (BOOL WINAPI (*)(HANDLE,int)) GetProcAddress(hModule, "AvSetMmThreadPriority");
        if(AvSetMmThreadPriority) {
          AvSetMmThreadPriority(hTask, 2 /*AVRT_PRIORITY_CRITICAL*/);
        }
      }
      FreeLibrary(hModule);
    }
  }

  bool threadinit(const HANDLE hEvent) {
    HRESULT result;

	CoInitializeEx(NULL, COINIT_MULTITHREADED | COINIT_SPEED_OVER_MEMORY);

    IMMDeviceEnumerator *pEnumerator;
    result = CoCreateInstance(CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, IID_IMMDeviceEnumerator, (void**)&pEnumerator);
    if(FAILED(result)) {
      logerr("Failed to activate MMDeviceEnumerator.");
      return false;
    }
    result = pEnumerator->GetDefaultAudioEndpoint(eRender, eConsole, &pDevice);
    pEnumerator->Release();
    if(FAILED(result)) {
      logerr("No audio endpoints found.");
      return false;
    }

    result = pDevice->Activate(IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pAudioClient);
    if(FAILED(result)) {
      logerr("Failed to activate AudioClient.");
      return false;
    }

    #ifdef EXCLUSIVE_MODE
      const WAVEFORMATEX wfx = {WAVE_FORMAT_PCM, 2, settings.frequency, settings.frequency*4, 4, 16, 0};
      result = pAudioClient->IsFormatSupported(AUDCLNT_SHAREMODE_EXCLUSIVE, &wfx, NULL);
      if(FAILED(result)) {
        logerr("The selected output format is not supported for exclusive mode on this audio hardware.");
        return false;
      }
    #else
      pAudioClient->GetMixFormat(&wfx); //There's no arguing. Shared mode can't change the system playback format.
      settings.frequency = wfx->nSamplesPerSec;
    #endif

    pAudioClient->GetDevicePeriod(
      #ifdef USE_SAFE_WASAPI_LATENCY
        reinterpret_cast<LONGLONG*>(&device.latency), NULL
      #else
        NULL, reinterpret_cast<LONGLONG*>(&device.latency)
      #endif
    );

    #ifdef FORCE_MIN_WASAPI_LATENCY
      if(device.latency < FORCE_MIN_WASAPI_LATENCY) {
        *const_cast<volatile uint64_t*>(&device.latency) = FORCE_MIN_WASAPI_LATENCY;
      }
    #endif

    result = pAudioClient->Initialize(
      #ifdef EXCLUSIVE_MODE
        AUDCLNT_SHAREMODE_EXCLUSIVE, AUDCLNT_STREAMFLAGS_EVENTCALLBACK, device.latency, device.latency, &wfx, NULL
      #else
        AUDCLNT_SHAREMODE_SHARED, AUDCLNT_STREAMFLAGS_EVENTCALLBACK, device.latency, 0, wfx, NULL
      #endif
    );
    if(FAILED(result)) {
      logerr("WASAPI failed to initialize.");
      return false;
    }

    device.localbuffers = (20000*settings.latency + device.latency)/(2*device.latency);
    if(!device.localbuffers) device.localbuffers++;

    pAudioClient->SetEventHandle(hEvent);

    pAudioClient->GetBufferSize(&device.bufferframes);
    *const_cast<volatile uint32_t**>(&device.localbuffer) = new uint32_t[device.bufferframes * device.localbuffers];
    if(!device.localbuffer) {
      #ifdef CONSOLE
        result = 0;
      #endif
      logerr("Memory could not be allocated for the ring buffer.");
      return false;
    }

    result = pAudioClient->GetService(IID_IAudioRenderClient, (void**)&pRenderClient);
    if(FAILED(result)) {
      logerr("Failed to activate RenderClient.");
      return false;
    }
    BYTE *endpointbuffer;
    result = pRenderClient->GetBuffer(device.bufferframes, &endpointbuffer);
    if(FAILED(result)) {
      logerr("RenderClient could not be started.");
      return false;
    }
    memset(endpointbuffer, 0, device.bufferframes*4);
    pRenderClient->ReleaseBuffer(device.bufferframes, 0);
    result = pAudioClient->Start();
    if(FAILED(result)) {
      logerr("AudioClient could not be started.");
      return false;
    }

    if(WaitForSingleObject(hEvent, 2000) /*Not signaled*/) {
      #ifdef CONSOLE
        result = 0;
      #endif
      logerr("Buffering events are not being dispatched.");
      return false;
    }

    return true;
  }

  #ifdef EXCLUSIVE_MODE
  bool reinit(const HANDLE hEvent) {
    HRESULT result;

    result = pDevice->Activate(IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pAudioClient);
    if(FAILED(result)) {
      logerr("Failed to activate AudioClient.");
      return false;
    }

    const WAVEFORMATEX wfx = {WAVE_FORMAT_PCM, 2, settings.frequency, settings.frequency*4, 4, 16, 0};
    result = pAudioClient->IsFormatSupported(AUDCLNT_SHAREMODE_EXCLUSIVE, &wfx, NULL);
    if(FAILED(result)) {
      logerr("The selected output format is not supported for exclusive mode on this audio hardware.");
      return false;
    }

    result = pAudioClient->Initialize(
      AUDCLNT_SHAREMODE_EXCLUSIVE, AUDCLNT_STREAMFLAGS_EVENTCALLBACK, device.latency, device.latency, &wfx, NULL
    );
    if(FAILED(result)) {
      logerr("WASAPI failed to initialize.");
      return false;
    }

    pAudioClient->SetEventHandle(hEvent);

    result = pAudioClient->GetService(IID_IAudioRenderClient, (void**)&pRenderClient);
    if(FAILED(result)) {
      logerr("Failed to activate RenderClient.");
      return false;
    }
    BYTE *endpointbuffer;
    result = pRenderClient->GetBuffer(device.bufferframes, &endpointbuffer);
    if(FAILED(result)) {
      logerr("RenderClient could not be started.");
      return false;
    }
    memset(endpointbuffer, 0, device.bufferframes*4);
    pRenderClient->ReleaseBuffer(device.bufferframes, 0);
    result = pAudioClient->Start();
    if(FAILED(result)) {
      logerr("AudioClient could not be started.");
      return false;
    }

    if(WaitForSingleObject(hEvent, 2000) /*Not signaled*/) {
      #ifdef CONSOLE
        result = 0;
      #endif
      logerr("Buffering events are not being dispatched.");
      return false;
    }

    return true;
  }
  #endif

  static DWORD WINAPI ThreadProc(pAudioWASAPI& __restrict__ obj) __attribute__((hot)) {
    const HANDLE hEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
    if(!obj.threadinit(hEvent)) {
      CloseHandle(hEvent);
      obj.Terminate = true;
      SetEvent(obj.BufferAvailable);
      return false;
    }
    obj.SetPriority(obj.hThread, true);
    SetEvent(obj.BufferAvailable);

    unsigned immediatebuffer = 0;

    while(!obj.Terminate) {
      #ifdef EXCLUSIVE_MODE
      if(obj.Pause) {
        BYTE * endpointbuffer;
        obj.pRenderClient->GetBuffer(obj.device.bufferframes, &endpointbuffer);
        obj.pRenderClient->ReleaseBuffer(obj.device.bufferframes, AUDCLNT_BUFFERFLAGS_SILENT);
        WaitForSingleObject(hEvent, INFINITE);

        obj.pAudioClient->Stop();
        obj.pAudioClient->Reset();
        obj.pRenderClient->Release();
        obj.pAudioClient->Release();
        obj.pAudioClient = 0;
        obj.pRenderClient = 0;

        do {
          obj.Pause = false;
          Sleep(60); //bsnes sends Pause pulse once per 20ms
        } while(obj.Pause == true);

        if(!obj.reinit(hEvent)) {
          CloseHandle(hEvent);
          return false;
        }
      }
      #endif

      BYTE * endpointbuffer;
      WaitForSingleObject(hEvent, INFINITE);
      obj.pRenderClient->GetBuffer(obj.device.bufferframes, &endpointbuffer);
      if(obj.device.buffersfilled) {
        memcpy(endpointbuffer, obj.device.localbuffer + immediatebuffer, obj.device.bufferframes<<2);
        obj.pRenderClient->ReleaseBuffer(obj.device.bufferframes, 0);

        EnterCriticalSection(&obj.ChangeBufferCount);
        obj.device.buffersfilled--;
        SetEvent(obj.BufferAvailable);
        LeaveCriticalSection(&obj.ChangeBufferCount);

        if((immediatebuffer+=obj.device.bufferframes) == obj.device.localbuffers*obj.device.bufferframes) immediatebuffer = 0;
      } else {
        obj.pRenderClient->ReleaseBuffer(obj.device.bufferframes, AUDCLNT_BUFFERFLAGS_SILENT);
      }
    }

    BYTE * endpointbuffer;
    obj.pRenderClient->GetBuffer(obj.device.bufferframes, &endpointbuffer);
    obj.pRenderClient->ReleaseBuffer(obj.device.bufferframes, AUDCLNT_BUFFERFLAGS_SILENT);
    WaitForSingleObject(hEvent, INFINITE);
    CloseHandle(hEvent);
    return true;
  }

  void term() {
    if(hThread) {
      Terminate = true;
      DWORD ExitCode;
      while(GetExitCodeThread(hThread, &ExitCode) == STILL_ACTIVE);
      CloseHandle(hThread);
      hThread = 0;
    }

    if(pRenderClient) {
      if(pAudioClient) {
        pAudioClient->Stop();
      }
      pRenderClient->Release();
      pRenderClient = 0;
    }
    if(pAudioClient) {
      pAudioClient->Release();
      pAudioClient = 0;
    }
    if(pDevice) {
      pDevice->Release();
      pDevice = 0;
    }
    if(wfx) {
      CoTaskMemFree(wfx);
      wfx = 0;
    }
    if(BufferAvailable) {
      CloseHandle(BufferAvailable);
      BufferAvailable = 0;
    }
    DeleteCriticalSection(&ChangeBufferCount);

    if(device.localbuffer) {
      delete[] device.localbuffer;
      device.localbuffer = 0;
    }

    Pause = false;
    Terminate = false;

    device.buffersfilled = 0;
    device.onbuffer = 0;
    device.frameoffset = 0;
  }

  pAudioWASAPI() {
    pAudioClient = 0;
    pRenderClient = 0;
    pDevice = 0;
    wfx = 0;
    BufferAvailable = 0;

    Pause = false;
    Terminate = false;

    device.localbuffer = 0;
    device.buffersfilled = 0;
    device.onbuffer = 0;
    device.frameoffset = 0;

    settings.synchronize = false;
    settings.frequency = 48000;
    settings.latency = 60;
  }
};

DeclareAudio(WASAPI)

}